package jeresources.profiling;

import jeresources.config.Settings;
import jeresources.json.ProfilingAdapter;
import jeresources.util.LogHelper;
import net.minecraft.client.Minecraft;
import net.minecraft.command.ICommandSender;
import net.minecraft.entity.Entity;
import net.minecraft.util.text.TextComponentString;
import net.minecraft.world.WorldServer;
import net.minecraftforge.common.DimensionManager;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

public class Profiler implements Runnable {
    private final ConcurrentMap<Integer, ProfiledDimensionData> allDimensionData;
    private final ProfilingTimer timer;
    private final ICommandSender sender;
    private final ProfilingBlacklist blacklist;
    private final int chunkCount;
    private final boolean allWorlds;
    private ProfilingExecutor currentExecutor;

    private Profiler(ICommandSender sender, int chunkCount, boolean allWorlds) {
        this.sender = sender;
        this.allDimensionData = new ConcurrentHashMap<>();
        this.chunkCount = chunkCount;
        this.timer = new ProfilingTimer(sender, chunkCount);
        this.allWorlds = allWorlds;
        this.blacklist = new ProfilingBlacklist();
    }

    @Override
    public void run() {
        LogHelper.warn("There will be messages about world gen lag during the profiling, you can ignore these as that is what you get when profiling.");
        if (!allWorlds) {

            // Will never be null as the mod is client side only
            Entity sendEnt = sender.func_174793_f();
            int dimId = sendEnt.field_71093_bK;
            profileWorld(dimId);
            
        } else {        

            //getStaticDimensionIDs gets ALL of the dimensions.
            //Forge says it is internal, but there are not other options for
            //all dimensions that exist.
            for (Integer dimId : DimensionManager.getStaticDimensionIDs()) {
                profileWorld(dimId);
            }
        }

        writeData();

        this.timer.complete();
    }

    private void profileWorld(int dimId) {
        String msg;
        //Get the world we want to process.
        WorldServer world = DimensionManager.getWorld(dimId);
        
        //If it isn't loaded, make it loaded and get it again.
        if(world == null) {
            DimensionManager.initDimension(dimId);
            world = DimensionManager.getWorld(dimId);
        }
        
        if (world == null) {
            msg = "Unable to profile dimension " + dimId + ".  There is no world for it.";
            LogHelper.error(msg);
            sender.func_145747_a(new TextComponentString(msg));
            return;
        }
        
        //Make this stick for recovery after profiling.
        final WorldServer worldServer = world;
        
        msg = "Inspecting dimension " + dimId + ": " + worldServer.field_73011_w.func_186058_p().func_186065_b() + ". ";
        sender.func_145747_a(new TextComponentString(msg));
        
        msg += "The world thinks it is dimension " + worldServer.field_73011_w.getDimension() + ".";
        LogHelper.info(msg);

        
        if (Settings.excludedDimensions.contains(dimId)) {
            msg = "Skipped dimension " + dimId + " during profiling";
            LogHelper.info(msg);
            sender.func_145747_a(new TextComponentString(msg));
            return;
        }

        final ProfilingExecutor executor = new ProfilingExecutor(this);
        this.currentExecutor = executor;
        this.allDimensionData.put(dimId, new ProfiledDimensionData());

        DummyWorld dummyWorld = new DummyWorld(worldServer);
        dummyWorld.func_175643_b();
        ChunkGetter chunkGetter = new ChunkGetter(chunkCount, dummyWorld, executor);
        worldServer.func_152344_a(chunkGetter);

        executor.awaitTermination();
        this.currentExecutor = null;

        dummyWorld.clearChunks();
        // Return the world to it's original state
        DimensionManager.setWorld(dimId, worldServer, Minecraft.func_71410_x().func_71401_C());
    }

    public ProfilingTimer getTimer() {
        return timer;
    }

    public ProfilingBlacklist getBlacklist() {
        return blacklist;
    }

    public ConcurrentMap<Integer, ProfiledDimensionData> getAllDimensionData() {
        return allDimensionData;
    }

    private void writeData() {
        Map<Integer, ProfilingAdapter.DimensionData> allData = new HashMap<>();
        for (Integer dim : this.allDimensionData.keySet()) {
            ProfiledDimensionData profiledData = this.allDimensionData.get(dim);
            ProfilingAdapter.DimensionData data = new ProfilingAdapter.DimensionData();
            data.dropsMap = profiledData.dropsMap;
            data.silkTouchMap = profiledData.silkTouchMap;

            for (Map.Entry<String, Integer[]> entry : profiledData.distributionMap.entrySet()) {
                Float[] array = new Float[ChunkProfiler.CHUNK_HEIGHT];
                for (int i = 0; i < ChunkProfiler.CHUNK_HEIGHT; i++)
                    array[i] = entry.getValue()[i] * 1.0F / this.timer.getBlocksPerLayer(dim);
                data.distribution.put(entry.getKey(), array);
            }

            allData.put(dim, data);
        }

        ProfilingAdapter.write(allData);
    }

    private static Profiler currentProfiler;

    public static boolean init(ICommandSender sender, int chunks, boolean allWorlds) {
        if (currentProfiler != null && !currentProfiler.timer.isCompleted()) return false;
        currentProfiler = new Profiler(sender, chunks, allWorlds);
        new Thread(currentProfiler).start();
        return true;
    }

    public static boolean stop() {
        if (currentProfiler == null || currentProfiler.timer.isCompleted()) return false;
        if (currentProfiler.currentExecutor != null)
            currentProfiler.currentExecutor.shutdownNow();
        currentProfiler.writeData();
        return true;
    }

}
